<?php
/**
 * Plugin Name: MPM Pre-Install Checker
 * Description: MPM (Marketing Property Manager) を設置する前に、サーバー環境が適合しているか（文字コード変換、メモリ、通信など）を単独で診断します。
 * Version: 2.0.0
 * Author: オレンジホールディング株式会社
 * Author URI: https://print55.net/
 * License: GPL2
 */

if ( ! defined( 'ABSPATH' ) ) exit;

class MPM_Pre_Check {
    public function __construct() {
        add_action('admin_menu', [$this, 'add_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('wp_ajax_mpm_test_simulation', [$this, 'test_import_simulation']);
        add_action('wp_ajax_mpm_test_ajax', [$this, 'test_ajax_connection']);
    }

    public function add_menu() {
        // add_management_page (ツール内) から add_menu_page (トップレベル) に変更
        add_menu_page(
            '予約管理設置前診断',      // ページタイトル
            '予約管理設置前診断',      // メニュー名称
            'manage_options',          // 権限
            'mpm-pre-check',           // スラッグ
            [$this, 'render_page'],    // コールバック関数
            'dashicons-clipboard'      // アイコン (クリップボード風)
        );
    }

    public function enqueue_scripts($hook) {
        // トップレベルメニューになったため、フック名の判定ロジックを変更
        if (strpos($hook, 'page_mpm-pre-check') === false) return;
        wp_enqueue_script('jquery');
    }

    public function render_page() {
        ?>
        <div class="wrap">
            <h1>予約管理プラグイン設置前診断</h1>
            <p>このサーバーで「予約管理システム」が正常に動作するか、インストール前にチェックします。</p>

            <div class="card" style="max-width: 100%; padding: 20px; margin-top: 20px;">
                <h2>1. サーバー基本要件チェック</h2>
                <table class="widefat striped">
                    <thead><tr><th>項目</th><th>現在の値</th><th>判定</th></tr></thead>
                    <tbody>
                        <?php $this->check_specs(); ?>
                    </tbody>
                </table>
            </div>

            <div class="card" style="max-width: 100%; padding: 20px; margin-top: 20px;">
                <h2>2. データインポート機能のシミュレーション</h2>
                <p>
                    初期設定では、Shift-JIS形式の住所データを大量に読み込み、UTF-8に変換して保存します。<br>
                    実際のファイルを使わず、仮想データをメモリ上で生成して<strong>「文字化けせずに変換できるか」</strong>テストします。
                </p>
                <button type="button" id="btn-sim" class="button button-primary">シミュレーション実行</button>
                <div id="res-sim" style="margin-top: 15px; padding: 10px; background: #f0f0f1; border: 1px solid #ccc; display: none;"></div>
            </div>

            <div class="card" style="max-width: 100%; padding: 20px; margin-top: 20px;">
                <h2>3. 通信・ファイル権限テスト</h2>
                <table class="widefat striped">
                    <tbody>
                        <tr>
                            <td><strong>AJAX通信 (管理画面操作)</strong><br>WAF等によるブロックがないか確認</td>
                            <td id="res-ajax">
                                <button type="button" id="btn-ajax" class="button button-small">テスト</button>
                            </td>
                        </tr>
                        <tr>
                            <td><strong>画像アップロード権限</strong><br><code>uploads</code> フォルダへの書き込み権限</td>
                            <td><?php $this->check_upload_dir(); ?></td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <script>
            jQuery(document).ready(function($) {
                // インポートシミュレーション
                $('#btn-sim').click(function() {
                    var $box = $('#res-sim');
                    $box.show().html('処理中... (仮想Shift-JISデータを生成・変換しています)');
                    $.post(ajaxurl, { action: 'mpm_test_simulation' }, function(res) {
                        if(res.success) {
                            $box.html(res.data);
                        } else {
                            $box.html('<strong style="color:red;">失敗:</strong> ' + res.data);
                        }
                    }).fail(function() { $box.html('<span style="color:red;">通信エラー (サーバーが応答しません)</span>'); });
                });

                // AJAXテスト
                $('#btn-ajax').click(function() {
                    var $td = $('#res-ajax');
                    $td.text('通信中...');
                    $.post(ajaxurl, { action: 'mpm_test_ajax' }, function(res) {
                        if(res.success) $td.html('<span style="color:green;font-weight:bold;">OK: 正常に通信できました</span>');
                        else $td.html('<span style="color:red;">エラー: ' + res.data + '</span>');
                    }).fail(function(xhr) {
                        $td.html('<span style="color:red;">失敗 (HTTP ' + xhr.status + ') - WAF設定を確認してください</span>');
                    });
                });
            });
            </script>
        </div>
        <?php
    }

    private function check_specs() {
        // PHP
        $ver = phpversion();
        $is_php_ok = version_compare($ver, '7.4', '>=');
        echo "<tr><td>PHPバージョン</td><td>{$ver}</td><td>" . ($is_php_ok ? '<span style="color:green">OK</span>' : '<span style="color:red">7.4以上推奨</span>') . "</td></tr>";

        // mbstring (最重要)
        $mb = extension_loaded('mbstring');
        echo "<tr><td><strong>mbstring拡張 (必須)</strong></td><td>" . ($mb ? '有効' : '無効') . "</td><td>" . ($mb ? '<span style="color:green">OK</span>' : '<span style="color:red;font-weight:bold;">NG (インポート不可)</span>') . "</td></tr>";

        // Memory
        $mem = ini_get('memory_limit');
        $mem_val = $this->return_bytes($mem);
        // 128MB以上ならOKとする
        $mem_ok = ($mem_val >= 128 * 1024 * 1024);
        echo "<tr><td>メモリ制限 (memory_limit)</td><td>{$mem}</td><td>" . ($mem_ok ? '<span style="color:green">OK</span>' : '<span style="color:orange">256M以上推奨</span>') . "</td></tr>";
        
        // Time Limit
        $time = ini_get('max_execution_time');
        echo "<tr><td>実行時間制限</td><td>{$time}秒</td><td>短い場合、インポート設定で「チャンクサイズ」を減らす必要があります。</td></tr>";
    }

    private function check_upload_dir() {
        $upload_dir = wp_upload_dir();
        if (wp_is_writable($upload_dir['basedir'])) {
            echo '<span style="color:green">OK (書き込み可能)</span>';
        } else {
            echo '<span style="color:red">NG (書き込み権限がありません)</span><br><small>' . esc_html($upload_dir['basedir']) . '</small>';
        }
    }

    // AJAX: インポートシミュレーション (仮想データ使用)
    public function test_import_simulation() {
        if (!extension_loaded('mbstring')) {
            wp_send_json_error('mbstring拡張が無効なため、テストを実行できません。');
        }

        // 1. テストデータ作成 (Shift-JIS)
        // 「テスト,住所,データ」という文字列をSJISに変換
        $original_utf8 = "テスト県,テスト市,テスト町";
        $sjis_data = mb_convert_encoding($original_utf8, 'SJIS-win', 'UTF-8');

        // 2. 正常にSJISになっているか確認 (UTF-8のままだと文字化けするはず)
        if ($sjis_data === $original_utf8) {
            // 変換されなかった場合（システム設定によってはあり得る）
            $msg = "警告: 文字コード変換が機能していない可能性があります (入出力が同一)。";
        }

        // 3. 逆変換テスト (MPMが行う処理と同じ手順)
        $converted_back = mb_convert_encoding($sjis_data, 'UTF-8', 'SJIS-win, UTF-8');

        if ($converted_back === $original_utf8) {
            $html = "<ul style='margin:0;'>";
            $html .= "<li><strong>仮想データ生成(SJIS):</strong> 成功</li>";
            $html .= "<li><strong>文字コード変換(SJIS→UTF-8):</strong> <span style='color:green;font-weight:bold;'>成功</span></li>";
            $html .= "<li><strong>判定:</strong> このサーバーで住所・駅データのインポートは正常に動作します。</li>";
            $html .= "</ul>";
            wp_send_json_success($html);
        } else {
            wp_send_json_error("文字コード変換に失敗しました。<br>元データ: $original_utf8<br>変換後: $converted_back");
        }
    }

    public function test_ajax_connection() {
        wp_send_json_success();
    }

    // ヘルパー: メモリサイズ単位変換
    private function return_bytes($val) {
        $val = trim($val);
        $last = strtolower($val[strlen($val)-1]);
        $val = (int)$val;
        switch($last) {
            case 'g': $val *= 1024;
            case 'm': $val *= 1024;
            case 'k': $val *= 1024;
        }
        return $val;
    }
}

new MPM_Pre_Check();